/**
 * Package Body: googleSitemap
 * Author: Niall McPhillips 
 * Company: Long Acre sàrl
 *
 * ARTICLES Sitemaps Package - Implementation
 * 
 * This package body implements the functionality for generating XML sitemaps for the ARTICLES application.
 * It follows the Sitemaps XML protocol as specified by sitemaps.org and includes Google News extensions.
 *
 * Author: Seafarers Team
 * Created: [Date]
 * Last Modified: [Date]
 * 
 * Dependencies:
 * - vw_articles view: Source view containing article data (article_id, last_modified, publication_date, title)
 * - APEX utilities for HTTP response handling (owa_util, htp, apex_util)
 * - Base URL constant from package specification
 * 
 * Implementation Notes:
 * - Outputs XML content directly to HTTP response
 * - Implements dynamic priority and change frequency based on article age
 * - Supports Google News sitemap extensions for enhanced search visibility
 * - Uses the base URL constant defined in the package specification
 */
create or replace package body googleSitemap as

   /**
    * Procedure: generateMainSitemap
    * 
    * Generates the main XML sitemap for public articles with Google News extensions.
    * Includes links to individual article pages with last modification dates, priority,
    * and change frequency information.
    * 
    * Implementation Details:
    * - Creates XML sitemap following sitemaps.org protocol v0.9
    * - Includes Google News sitemap extensions (xmlns:news)
    * - Uses base URL from package constant c_base_url
    * - Constructs article URLs using details page with article_id parameter
    * 
    * Business Rules:
    * - Articles modified within 15 days: daily frequency, priority 0.8
    * - Articles modified within 90 days: weekly frequency, priority 0.6  
    * - Older articles: monthly frequency, priority 0.4
    * 
    * Output Format:
    * - Content-Type: application/xml
    * - UTF-8 encoding
    * - Well-formed XML with proper namespaces
    * - Ordered by publication date (most recent first)
    * 
    * Dependencies:
    * - vw_articles view with columns: article_id, last_modified, publication_date, title
    * - Package constant c_base_url for URL construction
    * 
    * @return Outputs XML sitemap directly to HTTP response (no return value)
    */
   procedure generateMainSitemap is
      l_clob_output  clob;  -- Holds the generated XML sitemap content
   begin
      -- ========================================================================
      -- MAIN SITEMAP GENERATION LOGIC
      -- ========================================================================
      -- 
      -- Generate XML sitemap by aggregating article data from vw_articles view.
      -- The query creates a complete XML document with:
      -- 1. XML declaration with UTF-8 encoding
      -- 2. urlset root element with namespace declarations
      -- 3. Individual URL entries for each article
      -- 4. Google News extensions for enhanced search visibility
      --
      select '<?xml version="1.0" encoding="UTF-8"?>' || chr(10) ||
              xmlserialize(
                content
                xmlelement(
                      "urlset", 
                      -- Define XML namespaces for standard sitemap and Google News
                      xmlattributes('http://www.sitemaps.org/schemas/sitemap/0.9'    as "xmlns",
                                    'http://www.google.com/schemas/sitemap-news/0.9' as "xmlns:news"),
                      -- Aggregate all article entries into XML elements
                      xmlagg(
                        xmlelement("url",
                          -- Article URL construction using package constant
                          xmlelement("loc", googleSitemap.c_base_url 
                                            || 'details?p3_article_id=' 
                                            || a.article_id),
                          -- Last modification date in ISO format
                          xmlelement("lastmod", to_char(a.last_modified, 'YYYY-MM-DD')),
                          -- Dynamic change frequency based on article age
                          xmlelement("changefreq", case
                                                     when a.last_modified > sysdate-15 then 'daily'   -- Recent articles change frequently
                                                     when a.last_modified > sysdate-90 then 'weekly'  -- Moderately recent articles
                                                                                       else 'monthly' -- Older articles change less
                                                    end),
                          -- Dynamic priority based on article recency (0.0-1.0 scale)
                          xmlelement("priority", case
                                                     when a.last_modified > sysdate-15 then '0.8'  -- High priority for recent content
                                                     when a.last_modified > sysdate-90 then '0.6'  -- Medium priority
                                                                                       else '0.4'  -- Lower priority for older content
                                                    end),
                           -- Google News sitemap extensions for enhanced news visibility
                           xmlelement("news:news",
                             xmlelement("news:publication",
                               xmlelement("news:name",'The Very Best Daily News'),      -- Publication name
                               xmlelement("news:language",'en')                         -- Content language
                              ),  
                             xmlelement("news:publication_date", a.publication_date),   -- Original publication date
                             xmlelement("news:title", a.title)                         -- Article title
                            )
                          )
                       -- Order by most recently modified first for optimal crawling
                       order by a.last_modified desc 
                     )
                   )
                -- Format XML with proper indentation for readability
                as clob indent size = 2
              ) as xml_output
         into l_clob_output 
         from vw_articles a
        -- Final ordering by publication date (most recent first)
        order by a.publication_date desc;

     -- ========================================================================
     -- HTTP RESPONSE GENERATION
     -- ========================================================================
     -- 
     -- Configure HTTP response headers and output the XML sitemap content.
     -- This section handles:
     -- 1. Setting appropriate MIME type for XML content
     -- 2. Specifying content length for efficient client handling
     -- 3. Outputting the generated XML without HTML escaping
     --
     
     -- Set HTTP response content type to XML
     owa_util.mime_header('application/xml', false);
     
     -- Specify content length for client optimization
     htp.p('Content-Length: ' || dbms_lob.getlength(l_clob_output));
     
     -- Close HTTP headers
     owa_util.http_header_close;
     
     -- Output the XML content without HTML escaping (preserve XML structure)
     apex_util.prn(p_clob => l_clob_output, p_escape => false);

   end generateMainSitemap;

-- ============================================================================
-- END OF PACKAGE BODY
-- ============================================================================

end googleSitemap;
/